const wppconnect = require('@wppconnect-team/wppconnect');
const config = require('./config/config');
const database = require('./modules/database');
const translator = require('./modules/translator');
const messageHandler = require('./modules/messageHandler');
const notifications = require('./modules/notifications');
const businessHours = require('./modules/businessHours');

class JoshBot {
  constructor() {
    this.client = null;
    this.isReady = false;
  }

  /**
   * Inicializa o bot
   */
  async start() {
    console.log('╔═══════════════════════════════════════════════════════╗');
    console.log('║                                                       ║');
    console.log('║              🤖 WHATSDESK PRO 🤖                      ║');
    console.log('║                                                       ║');
    console.log('║     AI-Powered Multilingual WhatsApp CRM System       ║');
    console.log('║                                                       ║');
    console.log('╚═══════════════════════════════════════════════════════╝');
    console.log('');

    try {
      // Initialize database
      console.log('📊 Initializing database...');
      database.initialize();

      // Initialize translator
      console.log('🌍 Initializing Google Translate...');
      await translator.initialize();

      // Initialize WPP Connect
      console.log('📱 Initializing WPP Connect...');
      await this.initializeWPPConnect();

      console.log('');
      console.log('✅ Bot initialized successfully!');
      console.log('');
      
      try {
        this.printStatus();
      } catch (error) {
        console.error('⚠️ Error displaying status:', error.message);
        console.log('');
        console.log('💡 Run: node scripts/reset-database.js');
        console.log('   To recreate the database');
        console.log('');
      }

    } catch (error) {
      console.error('❌ Error initializing bot:', error?.message || error);
      
      if (error?.code === 'SQLITE_ERROR') {
        console.log('');
        console.log('💡 Possible database issue.');
        console.log('   Run: node scripts/reset-database.js');
        console.log('');
      } else if (error?.message?.includes('qrReadFail') || error?.message?.includes('Login Fail')) {
        console.log('');
        console.log('⚠️ QR Code was not scanned in time!');
        console.log('');
        console.log('💡 Solutions:');
        console.log('   1. Run again: npm start');
        console.log('   2. Scan QR Code faster (3 minutes limit)');
        console.log('   3. Make sure WhatsApp is open on your phone');
        console.log('   4. Use a stable internet connection');
        console.log('');
      }
      
      process.exit(1);
    }
  }

  /**
   * Pergunta método de login ao usuário
   */
  async askLoginMethod() {
    const readline = require('readline');
    const rl = readline.createInterface({
      input: process.stdin,
      output: process.stdout
    });

    return new Promise((resolve) => {
      console.log('');
      console.log('╔═══════════════════════════════════════════════════════╗');
      console.log('║        📱 CHOOSE AUTHENTICATION METHOD              ║');
      console.log('╚═══════════════════════════════════════════════════════╝');
      console.log('');
      console.log('1️⃣  QR Code (default - faster)');
      console.log('2️⃣  Phone Code via SMS/WhatsApp');
      console.log('');
      
      rl.question('Enter 1 or 2: ', (answer) => {
        rl.close();
        resolve(answer.trim() === '2' ? 'code' : 'qrcode');
      });
    });
  }

  /**
   * Pergunta número de telefone para login por código
   */
  async askPhoneNumber() {
    const readline = require('readline');
    const rl = readline.createInterface({
      input: process.stdin,
      output: process.stdout
    });

    return new Promise((resolve) => {
      console.log('');
      console.log('📱 Enter your phone number with country code');
      console.log('Example: 258840000000 (Mozambique)');
      console.log('         5521999999999 (Brazil)');
      console.log('         1234567890 (USA)');
      console.log('');
      
      rl.question('Phone number: ', (answer) => {
        rl.close();
        resolve(answer.trim());
      });
    });
  }

  /**
   * Inicializa WPP Connect
   */
  async initializeWPPConnect() {
    try {
      console.log('🔍 Checking for existing session...');
      
      const fs = require('fs');
      const path = require('path');
      const tokenPath = path.join(__dirname, 'tokens', config.bot.sessionName);
      
      // Verifica se existe sessão salva (pasta e arquivos de sessão)
      let hasSession = false;
      if (fs.existsSync(tokenPath)) {
        // Verifica se há arquivos de sessão dentro da pasta
        const sessionFiles = fs.readdirSync(tokenPath);
        hasSession = sessionFiles.length > 0;
        
        if (hasSession) {
          console.log('✅ Session found in tokens/ folder');
          console.log(`📂 Found ${sessionFiles.length} session files`);
          console.log('💡 No authentication required - using existing session');
        }
      }
      
      let loginMethod = null;
      let phoneNumber = null;
      
      if (!hasSession) {
        console.log('⚠️ No session found');
        console.log('');
        
        // Pergunta método de login
        loginMethod = await this.askLoginMethod();
        
        if (loginMethod === 'code') {
          phoneNumber = await this.askPhoneNumber();
          console.log('');
          console.log('✅ Phone number configured:', phoneNumber);
          console.log('📲 You will receive an 8-digit code on WhatsApp');
          console.log('');
        }
      }
      
      // Configurações base
      const createOptions = {
        session: config.bot.sessionName,
        folderNameToken: './tokens',
        mkdirFolderToken: true,
        headless: 'new',
        devtools: false,
        useChrome: true,
        debug: false,
        logQR: !hasSession && loginMethod === 'qrcode',
        autoClose: 0,
        disableSpins: true,
        disableWelcome: true,
        updatesLog: false,
        createPathFileToken: true,
        waitForLogin: true,
        disableWelcomeCheck: true,
        browserRevision: '',
        catchQR: (!hasSession && loginMethod === 'qrcode') ? (base64Qr, asciiQR) => {
          console.log('');
          console.log('╔═══════════════════════════════════════════════════════╗');
          console.log('║          📱 SCAN QR CODE WITH WHATSAPP              ║');
          console.log('╚═══════════════════════════════════════════════════════╝');
          console.log('');
          console.log(asciiQR);
          console.log('');
          console.log('⏰ You have 3 MINUTES to scan');
          console.log('💡 Open WhatsApp > Menu (⋮) > Linked Devices');
          console.log('💡 Or use the visual QR Code that opened in browser');
          console.log('');
        } : undefined,
        catchLinkCode: (!hasSession && loginMethod === 'code') ? (code) => {
          console.log('');
          console.log('╔═══════════════════════════════════════════════════════╗');
          console.log('║             🔐 AUTHENTICATION CODE                    ║');
          console.log('╚═══════════════════════════════════════════════════════╝');
          console.log('');
          console.log('📲 Code: ' + code);
          console.log('');
          console.log('💡 Enter this code on WhatsApp on your phone');
          console.log('   1. Open WhatsApp');
          console.log('   2. Tap More options > Linked devices');
          console.log('   3. Tap Link a device');
          console.log('   4. Tap "Link with phone number"');
          console.log('   5. Enter the code above');
          console.log('');
        } : undefined,
        phoneNumber: !hasSession ? phoneNumber : null,
        statusFind: (statusSession, session) => {
          console.log('');
          console.log('📊 Status:', statusSession);
          
          if (statusSession === 'isLogged') {
            console.log('✅ Saved session found! Reconnecting automatically...');
            console.log('💡 No need to authenticate again!');
            console.log('💾 Using session: ' + tokenPath);
          } else if (statusSession === 'qrReadSuccess') {
            console.log('✅ QR Code read successfully! Wait for sync...');
            console.log('⏳ Waiting for WhatsApp Web confirmation...');
          } else if (statusSession === 'notLogged') {
            if (hasSession) {
              console.log('⚠️ Session exists but not logged - will try to restore...');
            } else if (loginMethod === 'code') {
              console.log('📱 Waiting for authentication code...');
            } else if (loginMethod === 'qrcode') {
              console.log('📱 First connection - scan QR Code');
            }
          } else if (statusSession === 'desconnectedMobile') {
            console.log('⚠️ Phone disconnected - keep WhatsApp open');
          } else if (statusSession === 'deviceNotConnected') {
            console.log('⚠️ Device not connected - reconnecting...');
          } else if (statusSession === 'serverClose') {
            console.log('⚠️ Server disconnected - trying to reconnect...');
          } else if (statusSession === 'autocloseCalled') {
            console.log('⚠️ Authentication timeout');
          } else if (statusSession === 'inChat') {
            console.log('✅ Connected and ready! Session being saved automatically...');
            console.log('💾 Session saved at: ' + tokenPath);
          }
        },
        puppeteerOptions: {
          headless: 'new',
          userDataDir: tokenPath,
          args: [
            '--no-sandbox',
            '--disable-setuid-sandbox',
            '--disable-dev-shm-usage',
            '--disable-accelerated-2d-canvas',
            '--no-first-run',
            '--no-zygote',
            '--disable-gpu',
            '--disable-extensions',
            '--disable-background-timer-throttling',
            '--disable-backgrounding-occluded-windows',
            '--disable-renderer-backgrounding',
            '--disable-blink-features=AutomationControlled',
            '--user-data-dir=' + tokenPath
          ]
        }
      };
      
      this.client = await wppconnect.create(createOptions);

      // Define handlers
      this.setupMessageHandlers();

      // Define cliente nos módulos
      messageHandler.setClient(this.client);
      notifications.setClient(this.client);

      this.isReady = true;
      console.log('✅ WPP Connect connected and ready!');
      
      // Start phone connection monitoring (checks every 30s)
      console.log('📡 Starting connection monitoring...');
      this.client.startPhoneWatchdog(30000);
      console.log('✅ Monitoring active - checking connection every 30s');
      
    } catch (error) {
      console.log('');
      console.log('⚠️ Failed to connect to WhatsApp');
      console.log('');
      throw new Error('Authentication not completed or connection failed');
    }
  }

  /**
   * Configura handlers de mensagens
   */
  setupMessageHandlers() {
    // Mensagens recebidas
    this.client.onMessage(async (message) => {
      try {
        // Verifica se o client ainda está conectado
        if (!this.client || !this.isReady) {
          console.log('⚠️ Client not ready, skipping message');
          return;
        }

        // Ignora mensagens de grupos
        if (message.isGroupMsg) {
          return;
        }

        // Ignora mensagens próprias
        if (message.fromMe) {
          return;
        }

        // Ignora status do WhatsApp
        if (message.from === 'status@broadcast' || message.to === 'status@broadcast') {
          return;
        }

        // Ignora mensagens de broadcast
        if (message.from.includes('broadcast')) {
          return;
        }

        // Processa mensagem
        await messageHandler.processMessage(message);

      } catch (error) {
        console.error('❌ Error processing message:', error);
      }
    });

    // Connection state
    this.client.onStateChange((state) => {
      console.log('🔄 Connection state:', state);
      
      if (state === 'CONFLICT' || state === 'UNPAIRED') {
        console.log('⚠️ WhatsApp disconnected - check your phone');
      }
    });

    // Mensagens lidas (ACK)
    this.client.onAck((ack) => {
      // Pode ser usado para atualizar status de mensagens
      // console.log('ACK:', ack);
    });
  }

  /**
   * Exibe status do bot
   */
  printStatus() {
    const stats = database.getStats();
    
    console.log('╔═══════════════════════════════════════════════════════╗');
    console.log('║                    📊 BOT STATUS                      ║');
    console.log('╠═══════════════════════════════════════════════════════╣');
    console.log(`║  🤖 Name: ${config.bot.name.padEnd(45)} ║`);
    console.log(`║  🏢 Company: ${config.bot.company.padEnd(41)} ║`);
    console.log(`║  🌍 Timezone: ${config.businessHours.timezone.padEnd(37)} ║`);
    console.log(`║  ⏰ Hours: ${config.businessHours.start} - ${config.businessHours.end.padEnd(31)} ║`);
    console.log('╠═══════════════════════════════════════════════════════╣');
    console.log(`║  📋 Open Tickets: ${String(stats.openTickets).padEnd(32)} ║`);
    console.log(`║  ✅ Closed Tickets: ${String(stats.closedTickets).padEnd(30)} ║`);
    console.log(`║  👥 Total Clients: ${String(stats.totalClients).padEnd(31)} ║`);
    console.log(`║  💬 Total Messages: ${String(stats.totalMessages).padEnd(29)} ║`);
    console.log('╠═══════════════════════════════════════════════════════╣');
    console.log(`║  🕐 Business Hours: ${businessHours.isBusinessHours() ? '✅ OPEN    '.padEnd(26) : '❌ CLOSED  '.padEnd(26)} ║`);
    console.log('╚═══════════════════════════════════════════════════════╝');
    console.log('');
    console.log('💡 Available commands:');
    console.log('   - Clients: /open ticket, /send, /change language');
    console.log('   - Agents: /listar tickets, /atender, /fechar ticket, /transfer');
    console.log('');
    console.log('📱 Bot is waiting for messages...');
    console.log('');
  }

  /**
   * Atualiza status periodicamente
   */
  startStatusUpdater() {
    // Atualiza status a cada 5 minutos
    setInterval(() => {
      if (this.isReady) {
        console.clear();
        this.printStatus();
      }
    }, 5 * 60 * 1000);
  }

  /**
   * Envia notificações pendentes periodicamente
   */
  startNotificationChecker() {
    // Verifica notificações pendentes a cada minuto
    setInterval(async () => {
      if (this.isReady) {
        await notifications.sendPendingNotifications();
      }
    }, 60 * 1000);
  }

  /**
   * Encerra bot gracefully
   */
  async shutdown() {
    console.log('');
    console.log('⏹️  Shutting down bot...');

    // Clear notification timers
    notifications.clearAllTimers();

    // Close database
    database.close();

    // Close WPP client while keeping session
    if (this.client) {
      try {
        // Stop connection monitoring
        console.log('📡 Stopping connection monitoring...');
        this.client.stopPhoneWatchdog();
        
        console.log('💾 Saving WhatsApp session...');
        
        // Wait a moment to ensure session is synced
        await new Promise(resolve => setTimeout(resolve, 3000));
        
        // Close browser without deleting session
        await this.client.close();
        
        console.log('✅ Session saved successfully!');
        console.log('📂 Files saved in: ./tokens/' + config.bot.sessionName);
        console.log('💡 Next run will restore the session automatically');
      } catch (error) {
        console.error('⚠️ Error closing client:', error.message);
      }
    }

    console.log('✅ Bot shut down successfully!');
    process.exit(0);
  }
}

// Cria instância do bot
const bot = new JoshBot();

// Inicia o bot
bot.start().then(() => {
  // Inicia atualizador de status
  bot.startStatusUpdater();
  
  // Inicia verificador de notificações
  bot.startNotificationChecker();
});

// Handlers de encerramento
process.on('SIGINT', async () => {
  await bot.shutdown();
});

process.on('SIGTERM', async () => {
  await bot.shutdown();
});

process.on('uncaughtException', (error) => {
  console.error('❌ Uncaught exception:', error);
  // Não termina o processo - continua executando
});

process.on('unhandledRejection', (error) => {
  console.error('❌ Unhandled rejection:', error);
  // Ignora erros do Puppeteer sobre frames detached
  if (error && error.message && error.message.includes('detached Frame')) {
    console.log('⚠️ Puppeteer frame detached - ignoring (normal during reconnection)');
    return;
  }
  // Não termina o processo - continua executando
});

module.exports = bot;
